<?php

if (! defined('ABSPATH')) {
	exit;
}

class WC_Gateway_Chestmoney extends WC_Payment_Gateway
{

	/**
	 * Chestmoney API token.
	 *
	 * @var string
	 */
	public $api_token;

	/**
	 * Gateway disabled message
	 *
	 * @var string
	 */
	public $msg;

	/**
	 * Constructor
	 */
	public function __construct()
	{
		$this->id                 = 'chestmoney';
		$this->method_title       = __('Chestmoney', 'woo-chestmoney');
		$this->method_description = sprintf(__('Chestmoney provide merchants with the tools and services needed to accept crypto payments from their customers using any crypto coin of their choice. <a href="%1$s" target="_blank">Sign up</a> for a Chestmoney account, and <a href="%2$s" target="_blank">get your API token</a>.', 'woo-chestmoney'), 'https://www.chestmoney.com/app/register', 'https://www.chestmoney.com/app/api-tokens');
		$this->has_fields = true;

		// Load the form fields
		$this->init_form_fields();

		// Load the settings
		$this->init_settings();

		// Get setting values
		$this->title              = $this->get_option('title');
		$this->description        = $this->get_option('description');
		$this->enabled            = $this->get_option('enabled');
		$this->api_token = $this->get_option('api_token');

		// Hooks
		add_action('admin_enqueue_scripts', array($this, 'admin_scripts'));

		add_action('admin_notices', array($this, 'admin_notices'));
		add_action(
			'woocommerce_update_options_payment_gateways_' . $this->id,
			array(
				$this,
				'process_admin_options',
			)
		);

		// Payment listener/API hook.
		add_action('woocommerce_api_wc_gateway_chestmoney', array($this, 'verify_chestmoney_transaction'));

		// Webhook listener/API hook.
		add_action('woocommerce_api_wc_gateway_chestmoney_webhook', array($this, 'process_webhooks'));

		// Check if the gateway can be used.
		if (! $this->is_valid_for_use()) {
			$this->enabled = false;
		}
	}

	/**
	 * Check if this gateway is enabled and available in the user's country.
	 */
	public function is_valid_for_use()
	{

		if (! in_array(get_woocommerce_currency(), apply_filters('woocommerce_chestmoney_supported_currencies', array('USD', 'EUR', 'GBP')))) {

			$this->msg = sprintf(__('Chestmoney does not support your store currency. Kindly set it to USD (&#36;), EUR (&#8364;) and GBP (&#163;) <a href="%s">here</a>', 'woo-chestmoney'), admin_url('admin.php?page=wc-settings&tab=general'));

			return false;
		}

		return true;
	}

	/**
	 * Display chestmoney payment icon.
	 */
	public function get_icon()
	{
		$icon = '<img src="' . WC_HTTPS::force_https_url(plugins_url('assets/images/chestmoney-wc.png', WC_CHESTMONEY_MAIN_FILE)) . '" alt="Chestmoney Payment Options" />';

		return apply_filters('woocommerce_gateway_icon', $icon, $this->id);
	}

	/**
	 * Check if Chestmoney merchant details is filled.
	 */
	public function admin_notices()
	{

		if ($this->enabled == 'no') {
			return;
		}

		// Check required fields.
		if (! ($this->api_token)) {
			echo '<div class="error"><p>' . sprintf(__('Please enter your Chestmoney merchant details <a href="%s">here</a> to be able to use the Chestmoney WooCommerce plugin.', 'woo-chestmoney'), admin_url('admin.php?page=wc-settings&tab=checkout&section=chestmoney')) . '</p></div>';
			return;
		}
	}

	/**
	 * Check if Chestmoney gateway is enabled.
	 *
	 * @return bool
	 */
	public function is_available()
	{

		if ('yes' == $this->enabled) {

			if (! ($this->api_token)) {

				return false;
			}

			return true;
		}

		return false;
	}

	/**
	 * Admin Panel Options.
	 */
	public function admin_options()
	{

?>

		<h2><?php _e('Chestmoney', 'woo-chestmoney'); ?>
			<?php
			if (function_exists('wc_back_link')) {
				wc_back_link(__('Return to payments', 'woo-chestmoney'), admin_url('admin.php?page=wc-settings&tab=checkout'));
			}
			?>
		</h2>

		<?php

		if ($this->is_valid_for_use()) {

			echo '<table class="form-table">';
			$this->generate_settings_html();
			echo '</table>';
		} else {
		?>
			<div class="inline error">
				<p><strong><?php _e('Chestmoney Payment Gateway Disabled', 'woo-chestmoney'); ?></strong>: <?php echo $this->msg; ?></p>
			</div>

<?php
		}
	}

	/**
	 * Initialise Gateway Settings Form Fields.
	 */
	public function init_form_fields()
	{

		$form_fields = array(
			'enabled'                          => array(
				'title'       => __('Enable/Disable', 'woo-chestmoney'),
				'label'       => __('Enable Chestmoney', 'woo-chestmoney'),
				'type'        => 'checkbox',
				'description' => __('Enable Chestmoney as a payment option on the checkout page.', 'woo-chestmoney'),
				'default'     => 'no',
				'desc_tip'    => true,
			),
			'api_token'                  => array(
				'title'       => __('API Token', 'woo-chestmoney'),
				'type'        => 'password',
				'description' => __('Get a free <a href="https://www.chestmoney.com/app/api-tokens" target="_blank">API Token</a> from your Chestmoney account.', 'woo-chestmoney'),
				'default'     => '',
			),
			'title'                            => array(
				'title'       => __('Title', 'woo-chestmoney'),
				'type'        => 'text',
				'description' => __('This controls the payment method title which the user sees during checkout.', 'woo-chestmoney'),
				'default'     => __('Pay with Crypto', 'woo-chestmoney'),
				'desc_tip'    => true,
			),
			'description'                      => array(
				'title'       => __('Description', 'woo-chestmoney'),
				'type'        => 'textarea',
				'description' => __('This controls the payment method description which the user sees during checkout.', 'woo-chestmoney'),
				'default'     => __('Make payment using your favorite cryptocurrency like bitcoin, etc', 'woo-chestmoney'),
				'desc_tip'    => true,
			),
		);

		$this->form_fields = $form_fields;
	}

	/**
	 * Payment form on checkout page
	 */
	public function payment_fields()
	{

		if ($this->description) {
			echo wpautop(wptexturize($this->description));
		}

		if (! is_ssl()) {
			return;
		}
	}

	/**
	 * Load admin scripts.
	 */
	public function admin_scripts()
	{

		if ('woocommerce_page_wc-settings' !== get_current_screen()->id) {
			return;
		}

		$suffix = defined('SCRIPT_DEBUG') && SCRIPT_DEBUG ? '' : '.min';

		$chestmoney_admin_params = array(
			'plugin_url' => WC_CHESTMONEY_URL,
			'nonce'      => wp_create_nonce('chestmoney_verify_api_token'),
		);

		add_thickbox();

		wp_enqueue_script('wc_chestmoney_admin', plugins_url('assets/js/chestmoney-admin' . $suffix . '.js', WC_CHESTMONEY_MAIN_FILE), array(), WC_CHESTMONEY_VERSION, true);

		wp_localize_script('wc_chestmoney_admin', 'wc_chestmoney_admin_params', $chestmoney_admin_params);
	}

	/**
	 * Process the payment.
	 *
	 * @param int $order_id
	 *
	 * @return array|void
	 */
	public function process_payment($order_id)
	{
		$order        = wc_get_order($order_id);
		$txnref       = $order_id . '_' . time();
		$invoice_items = array();
		$subtotal = 0;
		$total_tax = 0;

		foreach ($order->get_items() as $item_id => $item) {
			if (! $item instanceof \WC_Order_Item_Product) {
				continue;
			}

			$product_name = $item->get_name();
			$quantity = $item->get_quantity();
			$line_total   = $item->get_total();
			$tax_amount   = $item->get_total_tax();
			$unit_cost = $quantity > 0 ? ($line_total / $quantity) : 0;
			$invoice_items[] = array(
				'item' => $product_name,
				'quantity' => $quantity,
				'unit_cost' => $unit_cost,
			);

			$subtotal   += $line_total;
			$total_tax  += $tax_amount;
		}

		$tax_percent = $subtotal > 0 ? round(($total_tax / $subtotal) * 100, 2) : 0;

		$discount_type = 'Fixed Amount';
		$discount = $order->get_total_discount();

		$metadata = array(
			'custom_fields' => $this->get_custom_fields($order_id),
			'cancel_action' => wc_get_cart_url(),
			'website_url' => get_bloginfo('url'),
			'reference' => $txnref,
		);

		$chestmoney_params = array(
			'currency'     => $order->get_currency(),
			'from' => get_bloginfo('name'),
			'invoice_number'     => (string) $order_id,
			'customer_email'        => $order->get_billing_email(),
			'redirect_link' => WC()->api_request_url('WC_Gateway_Chestmoney'),
			'webhook_url' => WC()->api_request_url('WC_Gateway_Chestmoney_Webhook'),
			'invoice_items' => $invoice_items,
			'tax_percent'      => $tax_percent,
			'discount_type'    => $discount_type,
			'discount'         => $discount,
			'metadata'    => $metadata,
		);

		$chestmoney_url = 'https://www.chestmoney.com/api/v1/invoices/create';

		$headers = array(
			'Authorization' => 'Bearer ' . $this->api_token,
			'Content-Type'  => 'application/json',
			'Accept'  => 'application/json',
		);

		$args = array(
			'headers' => $headers,
			'timeout' => 60,
			'body'    => json_encode($chestmoney_params),
			'sslverify' => true,
		);
		$request = wp_remote_post($chestmoney_url, $args);

		if (! is_wp_error($request) && 201 === wp_remote_retrieve_response_code($request)) {
			$chestmoney_response = json_decode(wp_remote_retrieve_body($request));

			return array(
				'result'   => 'success',
				'redirect' => $chestmoney_response->data->authorization_url,
			);
		} else {
			wc_add_notice(__('Unable to process payment try again', 'woo-chestmoney'), 'error');

			return;
		}
	}

	/**
	 * Verify Chestmoney payment.
	 */
	public function verify_chestmoney_transaction()
	{
		if (isset($_REQUEST['reference'])) {
			$chestmoney_txn_ref = sanitize_text_field($_REQUEST['reference']);
			$order_id = (int) $_REQUEST['order_id'];
		} else {
			$chestmoney_txn_ref = false;
			$order_id = false;
		}

		@ob_clean();

		if ($chestmoney_txn_ref && $order_id) {
			$chestmoney_response = $this->get_chestmoney_transaction($chestmoney_txn_ref);
			$order = wc_get_order($order_id);

			if (false !== $chestmoney_response) {

				if ('Completed' == $chestmoney_response->data->payment_status) {
					if (in_array($order->get_status(), array('processing', 'completed', 'on-hold'))) {

						wp_redirect($this->get_return_url($order));

						exit;
					}

					$order_total      = $order->get_total();
					$order_currency   = $order->get_currency();
					$currency_symbol  = get_woocommerce_currency_symbol($order_currency);
					$amount_paid      = $chestmoney_response->data->amount;

					// check if the amount paid is equal to the order amount.
					if ($amount_paid < absint($order_total)) {

						$order->update_status('on-hold', '');

						$order->add_meta_data('_transaction_id', $chestmoney_txn_ref, true);

						$notice      = sprintf(__('Thank you for shopping with us.%1$sYour payment transaction was successful, but the amount paid is not the same as the total order amount.%2$sYour order is currently on hold.%3$sKindly contact us for more information regarding your order and payment status.', 'woo-chestmoney'), '<br />', '<br />', '<br />');
						$notice_type = 'notice';

						// Add Customer Order Note
						$order->add_order_note($notice, 1);

						// Add Admin Order Note
						$admin_order_note = sprintf(__('<strong>Look into this order</strong>%1$sThis order is currently on hold.%2$sReason: Amount paid is less than the total order amount.%3$sAmount Paid was <strong>%4$s (%5$s)</strong> while the total order amount is <strong>%6$s (%7$s)</strong>%8$s<strong>Chestmoney Transaction Reference:</strong> %9$s', 'woo-chestmoney'), '<br />', '<br />', '<br />', $currency_symbol, $amount_paid, $currency_symbol, $order_total, '<br />', $chestmoney_txn_ref);
						$order->add_order_note($admin_order_note);

						function_exists('wc_reduce_stock_levels') ? wc_reduce_stock_levels($order_id) : $order->reduce_order_stock();

						wc_add_notice($notice, $notice_type);
					} else {

						$order->payment_complete($chestmoney_txn_ref);
						$order->add_order_note(sprintf(__('Payment via Chestmoney successful (Transaction Reference: %s)', 'woo-chestmoney'), $chestmoney_txn_ref));

						$order->update_status('completed');
					}

					$order->save();

					WC()->cart->empty_cart();
				} else {
					$order->update_status('failed', __('Payment was declined by Chestmoney.', 'woo-chestmoney'));
				}
			}

			wp_redirect($this->get_return_url($order));

			exit;
		}

		wp_redirect(wc_get_page_permalink('cart'));

		exit;
	}

	/**
	 * Process Webhook.
	 */
	public function process_webhooks()
	{

		if (! array_key_exists('HTTP_X_CHESTMONEY_SIGNATURE', $_SERVER) || (strtoupper($_SERVER['REQUEST_METHOD']) !== 'POST')) {
			exit;
		}

		$json = file_get_contents('php://input');

		// validate event do all at once to avoid timing attack.
		if ($_SERVER['HTTP_X_CHESTMONEY_SIGNATURE'] !== hash_hmac('sha512', $json, $this->api_token)) {
			exit;
		}

		$event = json_decode($json);

		if ('charge.success' !== strtolower($event->event)) {
			return;
		}

		sleep(10);

		$chestmoney_response = $this->get_chestmoney_transaction($event->data->reference);

		if (false === $chestmoney_response) {
			return;
		}

		$order_id = (int) $event->data->order_id;

		$order = wc_get_order($order_id);

		if (! $order) {
			return;
		}

		http_response_code(200);

		if (in_array(strtolower($order->get_status()), array('processing', 'completed', 'on-hold'), true)) {
			exit;
		}

		$order_currency = $order->get_currency();

		$currency_symbol = get_woocommerce_currency_symbol($order_currency);

		$order_total = $order->get_total();

		$amount_paid = $chestmoney_response->data->amount;

		$chestmoney_ref = $chestmoney_response->data->reference;

		// check if the amount paid is equal to the order amount.
		if ($amount_paid < absint($order_total)) {

			$order->update_status('on-hold', '');

			$order->add_meta_data('_transaction_id', $chestmoney_ref, true);

			$notice      = sprintf(__('Thank you for shopping with us.%1$sYour payment transaction was successful, but the amount paid is not the same as the total order amount.%2$sYour order is currently on hold.%3$sKindly contact us for more information regarding your order and payment status.', 'woo-chestmoney'), '<br />', '<br />', '<br />');
			$notice_type = 'notice';

			// Add Customer Order Note.
			$order->add_order_note($notice, 1);

			// Add Admin Order Note.
			$admin_order_note = sprintf(__('<strong>Look into this order</strong>%1$sThis order is currently on hold.%2$sReason: Amount paid is less than the total order amount.%3$sAmount Paid was <strong>%4$s (%5$s)</strong> while the total order amount is <strong>%6$s (%7$s)</strong>%8$s<strong>Chestmoney Transaction Reference:</strong> %9$s', 'woo-chestmoney'), '<br />', '<br />', '<br />', $currency_symbol, $amount_paid, $currency_symbol, $order_total, '<br />', $chestmoney_ref);
			$order->add_order_note($admin_order_note);

			function_exists('wc_reduce_stock_levels') ? wc_reduce_stock_levels($order_id) : $order->reduce_order_stock();

			wc_add_notice($notice, $notice_type);

			WC()->cart->empty_cart();
		} else {

			$order->payment_complete($chestmoney_ref);

			$order->add_order_note(sprintf(__('Payment via Chestmoney successful (Transaction Reference: %s)', 'woo-chestmoney'), $chestmoney_ref));

			WC()->cart->empty_cart();

			$order->update_status('completed');
		}

		$order->save();

		exit;
	}

	/**
	 * Get custom fields to pass to Chestmoney.
	 *
	 * @param int $order_id WC Order ID
	 *
	 * @return array
	 */
	public function get_custom_fields($order_id)
	{
		$order = wc_get_order($order_id);

		$custom_fields = array();

		$custom_fields['plugin'] = 'woo-chestmoney';
		$custom_fields['order_id'] = $order_id;
		$custom_fields['customer_name'] = $order->get_billing_first_name() . ' ' . $order->get_billing_last_name();
		$custom_fields['customer_email'] = $order->get_billing_email();
		$custom_fields['customer_phone'] = $order->get_billing_phone();

		$billing_address = $order->get_formatted_billing_address();
		$billing_address = esc_html(preg_replace('#<br\s*/?>#i', ', ', $billing_address));

		$custom_fields['billing_address'] = $billing_address;

		$shipping_address = $order->get_formatted_shipping_address();
		$shipping_address = esc_html(preg_replace('#<br\s*/?>#i', ', ', $shipping_address));
		if (empty($shipping_address)) {

			$billing_address = $order->get_formatted_billing_address();
			$billing_address = esc_html(preg_replace('#<br\s*/?>#i', ', ', $billing_address));

			$shipping_address = $billing_address;
		}
		$custom_fields['shipping_address'] = $shipping_address;

		return $custom_fields;
	}

	/**
	 * Checks if WC version is less than passed in version.
	 *
	 * @param string $version Version to check against.
	 *
	 * @return bool
	 */
	public function is_wc_lt($version)
	{
		if (defined('WC_VERSION')) {
			return version_compare(WC_VERSION, $version, '<');
		}

		return false;
	}

	/**
	 * Retrieve a transaction from Chestmoney.
	 *
	 * @param $chestmoney_txn_ref
	 * @return false|mixed
	 */
	private function get_chestmoney_transaction($chestmoney_txn_ref)
	{
		$chestmoney_url = add_query_arg(
			'reference',
			$chestmoney_txn_ref,
			'https://www.chestmoney.com/api/v1/transactions/verify'
		);

		$headers = array(
			'Authorization' => 'Bearer ' . $this->api_token,
			'Content-Type'  => 'application/json',
			'Accept'  => 'application/json',
		);

		$args = array(
			'headers' => $headers,
			'timeout' => 60,
			'sslverify' => true,
		);

		$request = wp_remote_get($chestmoney_url, $args);

		if (! is_wp_error($request) && 200 === wp_remote_retrieve_response_code($request)) {
			return json_decode(wp_remote_retrieve_body($request));
		}

		return false;
	}

	/**
	 * Get Chestmoney payment icon URL.
	 */
	public function get_logo_url()
	{
		$url = WC_HTTPS::force_https_url(plugins_url('assets/images/chestmoney-wc.png', WC_CHESTMONEY_MAIN_FILE));

		return apply_filters('wc_chestmoney_gateway_icon_url', $url, $this->id);
	}
}
